Option Explicit

'**********************************************************
'   1993 -  Gary Garrison
'           Software Assist Corporation
'**********************************************************

Type Scroll_Bar_Attributes
hWnd                As Integer      ' Handle of scrollbar
InUse               As Integer      ' Flag for table entry
InternalChange      As Integer      ' Internal change flag
LastTrueValue       As Long         ' Last true value
LastValue           As Integer      ' Last value of scroll bar
Counter             As Long         ' Incrementing counter
Factor              As Long         ' Value of each scroll value
TrueMax             As Long         ' True maximum for scrollbar
End Type

Dim SBA()           As Scroll_Bar_Attributes
Dim SBA_Is_Dimed    As Integer      ' Flag indicating SBA dimed
Dim iSBA            As Integer      ' Common index for SBA()

Function GetScrollBarChange (vsbObj As Control) As Long
'****************************************************************
'   Return the amount of the last change.
'****************************************************************

    GetScrollBarChange = 0
    iSBA = LocateScrollBar(vsbObj)
    If iSBA = 0 Then Exit Function
    If SBA(iSBA).Factor <> 1 Then
        GetScrollBarChange = (vsbObj.Value - 1) * SBA(iSBA).Factor + SBA(iSBA).Counter - SBA(iSBA).LastTrueValue
    Else
        GetScrollBarChange = vsbObj.Value - SBA(iSBA).LastTrueValue
    End If
End Function

Function GetScrollBarValue (vsbObj As Control) As Long
'****************************************************************
'   Get the current, true value of a scroll bar.
'****************************************************************

    GetScrollBarValue = 0
    iSBA = LocateScrollBar(vsbObj)
    If iSBA = 0 Then Exit Function
    If SBA(iSBA).Factor <> 1 Then
        GetScrollBarValue = (vsbObj.Value - 1) * SBA(iSBA).Factor + SBA(iSBA).Counter
    Else
        GetScrollBarValue = vsbObj.Value
    End If
End Function

Sub InitScrollBar (vsbObj As Control, MaxValue As Long)
'****************************************************************
'   Initialize a scrollbar.
'****************************************************************

    Dim i           As Integer
    Dim hWnd        As Integer

'****************************************************************
'   Either find an existing entry for the scrollbar or create
'   a new one.
'****************************************************************
    hWnd = vsbObj.hWnd
    iSBA = LocateScrollBar(vsbObj)
    If iSBA = 0 Then
        If Not SBA_Is_Dimed Then
            ReDim SBA(1 To 1) As Scroll_Bar_Attributes
            SBA_Is_Dimed = True
        End If
        For i = 1 To UBound(SBA)
            If SBA(i).hWnd = hWnd Then
                iSBA = i
            ElseIf Not SBA(i).InUse And iSBA = 0 Then
                iSBA = i
            End If
        Next i
    End If
    If iSBA = 0 Then
        ReDim Preserve SBA(i To UBound(SBA) + 1) As Scroll_Bar_Attributes
        iSBA = UBound(SBA)
    End If

'****************************************************************
'   Set the initial values for the scrollbar.
'****************************************************************
    SBA(iSBA).InUse = True
    SBA(iSBA).hWnd = hWnd
    If vsbObj.Value <> 1 Then SBA(iSBA).InternalChange = True
    SBA(iSBA).TrueMax = MaxValue
    SBA(iSBA).LastValue = 1
    SBA(iSBA).LastTrueValue = 1
    SBA(iSBA).Counter = 1
    
'****************************************************************
'   If the maximum value is greater than the range of a scrollbar
'   .MAX, create a factor for the value of each scrollbar varlue.
'   Otherwise, just treate it as a normal scrollbar.
'****************************************************************
    If MaxValue > 32767 Then
        SBA(iSBA).Factor = Int(Sqr(MaxValue))
        vsbObj.Max = SBA(iSBA).Factor + 3
        vsbObj.Min = 0
    Else
        SBA(iSBA).Factor = 1
        vsbObj.Max = MaxValue
        vsbObj.Min = 1
    End If

    vsbObj.Value = 1
End Sub

Function LocateScrollBar (vsbObj As Control) As Integer
'****************************************************************
'   Locate a scrollbar in the SBA(). If it does not exist,
'   return a 0.
'****************************************************************

    Dim i As Integer
    Dim hWnd As Integer
    LocateScrollBar = 0
    If Not SBA_Is_Dimed Then Exit Function
    hWnd = vsbObj.hWnd
    For i = 1 To UBound(SBA)
        If hWnd = SBA(i).hWnd Then
            LocateScrollBar = i
            Exit Function
        End If
    Next i
End Function

Function ScrollBarChangeEvent (vsbObj As Control) As Integer
'****************************************************************
'   Register a scrollbar change. Typically called by the
'   scrollbar's _Change event.
'
'   If this is an externally (hitting scroll bar) generated
'   event, True is returned. Otherwise, False is returned.
'****************************************************************

    Dim ChgAmt As Integer
    
    ScrollBarChangeEvent = False
'****************************************************************
'   Locate the scrollbar in the SBA(). If not found, just exit.
'****************************************************************
    iSBA = LocateScrollBar(vsbObj)
    If iSBA = 0 Then Exit Function

'****************************************************************
'   If being called by an internal change to the value, just
'   reset the InternalChange flag and exit.
'****************************************************************
    If SBA(iSBA).InternalChange Then SBA(iSBA).InternalChange = False: GoTo ScrollBarChangeEventExit
    SBA(iSBA).InternalChange = True
    ScrollBarChangeEvent = True
    
'****************************************************************
'   If the factor is 1, this is treated like a normal scrollbar.
'****************************************************************
    If SBA(iSBA).Factor = 1 Then
        SBA(iSBA).Counter = 1
        SBA(iSBA).LastTrueValue = SBA(iSBA).LastValue
        SBA(iSBA).LastValue = vsbObj.Value
        SBA(iSBA).InternalChange = False
        GoTo ScrollBarChangeEventExit
    End If

'****************************************************************
'   Record the LastTrueValue so the amount of change can be
'   determined externally.
'****************************************************************
    SBA(iSBA).LastTrueValue = (SBA(iSBA).LastValue - 1) * SBA(iSBA).Factor + SBA(iSBA).Counter

'****************************************************************
'   Determine the amount of change to the scrollbar and
'   increment/decrement the counter.
'****************************************************************
    ChgAmt = -(SBA(iSBA).LastValue - vsbObj.Value)
    SBA(iSBA).Counter = SBA(iSBA).Counter + ChgAmt

'****************************************************************
'   Cannot let the value go to 0 (Min) or we could never reach
'   scroll values less than 1 factor.
'****************************************************************
    If SBA(iSBA).Counter < 1 And vsbObj.Value < 1 Then
        SBA(iSBA).Counter = 1
        vsbObj.Value = 1
'****************************************************************
'   If the ChgAmt is equal to 1, we just have to see if counter
'   has gone negative in which case it needs to be set to
'   Factor-1, or if it has equaled the value of factor in which
'   case it is set to 0, or if the counter is within range in
'   which case we have to put the scrollbar's value back.
'****************************************************************
    ElseIf Abs(ChgAmt) = 1 Then
        If SBA(iSBA).Counter < 0 Then
            SBA(iSBA).Counter = SBA(iSBA).Factor - 1
        ElseIf SBA(iSBA).Counter = SBA(iSBA).Factor Then
            SBA(iSBA).Counter = 0
        Else
            vsbObj.Value = vsbObj.Value - ChgAmt
        End If
    Else
'****************************************************************
'   If the ChgAmt was not equal to 1, that means it was a major
'   move. Just change the counter to 0 and let the scrollbar's
'   value represent the true value.
'****************************************************************
        SBA(iSBA).Counter = 0
    End If
    
'****************************************************************
'   Record the value so that next time in we know what the
'   change amount is. Turn off the internal change flag.
'****************************************************************
    SBA(iSBA).LastValue = vsbObj.Value
    SBA(iSBA).InternalChange = False

'****************************************************************
'   Routine exit point. Check to make sure we have not gone
'   beyond the true maximum for the scrollbar. If so, set the
'   values to the maximum.
'****************************************************************
ScrollBarChangeEventExit:
    If (vsbObj.Value - 1) * SBA(iSBA).Factor + SBA(iSBA).Counter > SBA(iSBA).TrueMax Then
        SBA(iSBA).InternalChange = True
        SBA(iSBA).Counter = SBA(iSBA).TrueMax - ((SBA(iSBA).TrueMax \ SBA(iSBA).Factor) * SBA(iSBA).Factor)
        vsbObj.Value = (SBA(iSBA).TrueMax \ SBA(iSBA).Factor) + 1
        SBA(iSBA).LastValue = vsbObj.Value
    End If
End Function

Sub ScrollBarScrollEvent (vsbObj As Control)
'****************************************************************
'   Someone is tugging on the scrollbar's thumb.
'****************************************************************

    iSBA = LocateScrollBar(vsbObj)
    If iSBA = 0 Then Exit Sub

'****************************************************************
'   If factor is 1, this is just a normal scrollbar.
'****************************************************************
    If SBA(iSBA).Factor = 1 Then Exit Sub
    
'****************************************************************
'   Check to make sure don't go below 1 or above the maximum
'   value for the scrollbar.
'****************************************************************
    If vsbObj.Value = 0 Then
        SBA(iSBA).InternalChange = True
        SBA(iSBA).Counter = 1
        vsbObj.Value = 1
    ElseIf vsbObj.Value = vsbObj.Max Then
        SBA(iSBA).Counter = SBA(iSBA).TrueMax - ((vsbObj.Max - 2) * SBA(iSBA).Factor)
    End If
End Sub

Sub SetScrollBarValue (vsbObj As Control, newVal As Long)
'****************************************************************
'   Set the current, true value of a scroll bar.
'****************************************************************
    Dim sbVal As Long

    iSBA = LocateScrollBar(vsbObj)
    If iSBA = 0 Then Exit Sub
    
'****************************************************************
'   Make sure we are not going outside the valid range.
'****************************************************************
    If newVal < 1 Or newVal > SBA(iSBA).TrueMax Then Exit Sub

    SBA(iSBA).LastTrueValue = (vsbObj.Value - 1) * SBA(iSBA).Factor + SBA(iSBA).Counter
'****************************************************************
'   If the factor is 1, just set the value.
'****************************************************************
    If SBA(iSBA).Factor = 1 Then
        sbVal = newVal
    Else
        sbVal = (newVal \ SBA(iSBA).Factor) + 1
        SBA(iSBA).Counter = newVal - ((sbVal - 1) * SBA(iSBA).Factor)
    End If

'****************************************************************
'   If sbVal=scrollbar.Value, don't set InternalChange flag
'   since setting the value would not cause a change.
'****************************************************************
    SBA(iSBA).LastValue = sbVal
    If sbVal <> vsbObj.Value Then
        SBA(iSBA).InternalChange = True
        vsbObj.Value = sbVal
    End If
End Sub

Sub TermScrollBar (vsbObj As Control)
'****************************************************************
'   Teminate control of a scroll bar. This is not necessary at
'   the end of the application. It's just here to be neat.
'****************************************************************

    iSBA = LocateScrollBar(vsbObj)
    If iSBA = 0 Then Exit Sub

    SBA(iSBA).InUse = False
    SBA(iSBA).hWnd = 0
End Sub

